<?php

namespace App\Controllers;

use App\Controllers\BaseController;
use App\Libraries\Asaas;
use App\Libraries\Zdg;
use App\Models\ClienteModel;
use App\Models\ClienteDetalheModel;
use App\Models\ClientePedidoModel;
use App\Models\ContabilModel;
use App\Models\FaturaModel;
use App\Models\OrcamentoModel;
use App\Models\OrganizacaoModel;
use App\Models\ClienteZproModel;
use DateTime;
use DateTimeZone;

class Cliente extends BaseController
{


    /**
     * INDEX } Retorna listagem de todos os clientes
     * 
     * @param NULL
     * @return string
     */
    public function index() : string {

        
        // lista todos os clientes cadastrados
        $clienteModel = new ClienteModel();
        
        // filtra o tipo de cliente listado
        if($this->request->getGet('status') === '0'):
            // inativo
            $clienteModel->where('cli_ativo','0');
        elseif($this->request->getGet('status') === '2'):
            // excluído
            $clienteModel->onlyDeleted();
        else:
            // ativo
            $clienteModel->where('cli_ativo','1');
        endif;
        
        // define informacoes da pagina
        $dados = [
            'TituloPagina'      => 'Clientes | Página principal',
            'listagemClientes'  => $clienteModel->paginate(10),
            'pager'             => $clienteModel->pager,
        ];

        return view('admin/cliente/cliente-index', $dados);
    }



    /**
     * CADASTRA} Cadastra um novo cliente
     * 
     * @param NULL
     * @return array
     */
    public function cadastra() {
        
        // verifica se veio informação por post
        if ($this->request->getMethod() === 'post') {

            // cria regras de formulario
            $rules = [
                'input_name' =>[
                    'rules'     => 'required',
                    'errors'    => [
                        'required'  => 'Insira um nome de usuário.',
                    ],
                ],
                'input_email' => [
                    'rules' => 'required|valid_email',
                    'errors'    => [
                        'required'  =>  'Insira um e-mail de usuário.',
                        'valid_email' => 'Insira um e-mail válido.',
                    ],
                ],
                'input_pass' => [
                    'rules' => 'required|min_length[6]',
                    'errors'    => [
                        'required'  => 'Insira uma senha.',
                        'min_length' => 'A senha precisa ter pelo menos 6 caracteres.',
                    ],
                ],
                'input_pass_confirm' => [
                    'rules' => 'required|matches[input_pass]',
                    'errors'    => [
                        'required'  => 'Insira novamente a senha.',
                        'matches'   => 'As senhas precisam ser iguais para confirmar.',
                    ],
                ],
            ];

            // Validação apresentou erro
            if(!$this->validate($rules)){

                // recarrega a página
                return redirect()->back()->withInput();

            // Validação executou com sucesso
            }else{

                // agrupa os dados
                $formulario = $this->request->getPost();
                $formulario['input_pass'] = password_hash($formulario['input_pass'],PASSWORD_DEFAULT);
                $dados = [
                    'cli_nome'             => $formulario['input_name'],
                    'cli_email'            => $formulario['input_email'],
                    'cli_email_extra'      => $formulario['input_email_extra'],
                    'cli_telefone'         => $formulario['input_phone'],
                    'cli_cpfcnpj'          => $formulario['input_cpfcnpj'],
                    'cli_insc_estadual'    => $formulario['input_insc_estadual'],
                    'cli_endereco'         => $formulario['input_endereco'],
                    'cli_endereco_numero'  => $formulario['input_endereco_numero'],
                    'cli_endereco_compl'   => $formulario['input_endereco_compl'],
                    'cli_bairro'           => $formulario['input_bairro'],
                    'cli_municipio'        => $formulario['input_municipio'],
                    'cli_estado_uf'        => $formulario['input_estado_uf'],
                    'cli_pais'             => $formulario['input_pais'],
                    'cli_cep'              => $formulario['input_cep'],
                    'cli_senha'            => $formulario['input_pass'],
                    'cli_avatar'           => $formulario['input_avatar'],
                ];

                // insere na base de dados
                $clienteModel = new ClienteModel();

                // insere na base : apresentou erro
                if(!$clienteModel->save($dados)):
                    // recarrega a página
                    session()->setFlashdata('msgType', 'error');
                    session()->setFlashdata('msg', 'Erro no cadastro do Cliente.');
                    return redirect()->back()->withInput();

                //  insere na base : executou com sucesso
                else:
                    // redireciona a página
                    session()->setFlashdata('msgType', 'success');
                    session()->setFlashdata('msg', 'Cliente Cadastrado com Sucesso.');
                    return redirect()->to('/cliente/cadastra');
                endif;
            }
        }

        // continua se não veio por post

        // define informacoes da pagina
        $dados = [
            'TituloPagina'  =>  'Usuários | Cadastre um novo cliente',
        ];

        // constroi pagina
        return view('admin/cliente/cliente-cadastra', $dados);
    }



    /**
     * EDITA} Mostra as informações de um cliente e edita
     * 
     * @param int $id_cliente
     * @return string
     */
    public function edita(int $id_cliente) {

        // se veio informação por post
        if ($this->request->getMethod() === 'post') {

            // cria regras de formulario
            $rules = [
                'input_name' =>[
                    'rules'     => 'required',
                    'errors'    => [
                        'required'  => 'Insira um nome de usuário.',
                    ],
                ],
                'input_email' => [
                    'rules' => 'required|valid_email',
                    'errors'    => [
                        'required'  =>  'Insira um e-mail de usuário.',
                        'valid_email' => 'Insira um e-mail válido.',
                    ],
                ],
            ];
            // adiciona regas de formulario para senha
            $rules_pass = [
                'input_pass' => [
                    'rules' => 'required|min_length[6]',
                    'errors'    => [
                        'required'  => 'Insira uma senha.',
                        'min_length' => 'A senha precisa ter pelo menos 6 caracteres.',
                    ],
                ],
                'input_pass_confirm' => [
                    'rules' => 'required|matches[input_pass]',
                    'errors'    => [
                        'required'  => 'Insira novamente a senha.',
                        'matches'   => 'As senhas precisam ser iguais para confirmar.',
                    ],
                ],
            ];

            //se vier senha, adiciona para validate
            if($this->request->getPost('input_pass') != ''):
                $rules = array_merge($rules, $rules_pass);
            endif;

            // Validação apresentou erro
            if(!$this->validate($rules)){

                // recarrega a página
                session()->setFlashdata('msgType','warning');
                session()->setFlashdata('msg','Não validou');
                return redirect()->back()->withInput();

            // Validação executou com sucesso
            }else{

                // recebe avatar
                $file = $this->request->getFile('input_avatar');
                if($file!='' && $file->isValid() && !$file->hasMoved()):
                    if($file->move('public\uploads\avatar', $file->getRandomName())):
                        $filePath = 'uploads/avatar/'.$file->getName();
                    else:
                        session()->setFlashdata('msgType','warning');
                        session()->setFlashdata('msg',$file->getErrorString());
                        return redirect()->to(current_url());
                    endif;
                else:
                    $filePath = $this->request->getPost('cli_avatar_atual');
                endif;

                // recebe os dados
                $formulario = $this->request->getPost();
                //se vier com ou sem senha
                if($formulario['input_pass'] != ''):
                    $formulario['input_pass'] = password_hash($formulario['input_pass'],PASSWORD_DEFAULT);    
                    // encapsula dados com senha
                    $dados = [
                        'cli_nome'             => $formulario['input_name'],
                        'cli_email'            => $formulario['input_email'],
                        'cli_email_extra'      => $formulario['input_email_extra'],
                        'cli_telefone'         => $formulario['input_phone'],
                        'cli_cpfcnpj'          => $formulario['input_cpfcnpj'],
                        'cli_insc_estadual'    => $formulario['input_insc_estadual'],
                        'cli_endereco'         => $formulario['input_endereco'],
                        'cli_endereco_numero'  => $formulario['input_endereco_numero'],
                        'cli_endereco_compl'   => $formulario['input_endereco_compl'],
                        'cli_bairro'           => $formulario['input_bairro'],
                        'cli_municipio'        => $formulario['input_municipio'],
                        'cli_estado_uf'        => $formulario['input_estado_uf'],
                        'cli_pais'             => $formulario['input_pais'],
                        'cli_cep'              => $formulario['input_cep'],
                        'cli_senha'            => $formulario['input_pass'],
                        'cli_ativo'            => $formulario['input_ativo'],
                        'cli_avatar'           => $filePath,
                    ];
                else:
                    // encapsula dados sem senha
                    $dados = [
                        'cli_nome'             => $formulario['input_name'],
                        'cli_email'            => $formulario['input_email'],
                        'cli_email_extra'      => $formulario['input_email_extra'],
                        'cli_telefone'         => $formulario['input_phone'],
                        'cli_cpfcnpj'          => $formulario['input_cpfcnpj'],
                        'cli_insc_estadual'    => $formulario['input_insc_estadual'],
                        'cli_endereco'         => $formulario['input_endereco'],
                        'cli_endereco_numero'  => $formulario['input_endereco_numero'],
                        'cli_endereco_compl'   => $formulario['input_endereco_compl'],
                        'cli_bairro'           => $formulario['input_bairro'],
                        'cli_municipio'        => $formulario['input_municipio'],
                        'cli_estado_uf'        => $formulario['input_estado_uf'],
                        'cli_pais'             => $formulario['input_pais'],
                        'cli_cep'              => $formulario['input_cep'],
                        'cli_ativo'            => $formulario['input_ativo'],
                        'cli_avatar'           => $filePath,
                    ];
                endif;

                // Atualiza na base de dados
                $clienteModel = new ClienteModel();

                // Atualiza na base : apresentou erro
                if(!$clienteModel->update($formulario['id_cliente'], $dados)):
                    // recarrega a página
                    session()->setFlashdata('msgType', 'error');
                    session()->setFlashdata('msg', 'Esta atualização apresentou erro.');
                    return redirect()->back()->withInput();

                //  Atualiza na base : executou com sucesso
                else:
                    // redireciona a página
                    session()->setFlashdata('msgType', 'success');
                    session()->setFlashdata('msg', 'Cliente Atualizado com sucesso.');
                    return redirect()->to('/cliente/painel/'.$formulario['id_cliente']);
                endif;
            }
        }
        
        // continua se não veio por post

        // retorna todos os clientes cadastrados
        $clienteModel = new ClienteModel();

        // define informacoes da pagina
        $dados = [
            'TituloPagina'  => 'Usuários | Edite seus dados',
            'cliente'       => $clienteModel->find($id_cliente),
        ];

        return view('admin/cliente/cliente-edita', $dados);
    }



    /**
     * EXCLUI} Remove as informações de um cliente da base de dados
     * 
     * @param int $id_cliente
     * @return string
     */
    public function exclui(int $id_cliente) {

        // Exclui usuário        
        $clienteModel = new ClienteModel();
        $confirma = $clienteModel->delete($id_cliente);

        // excluir cliente retornou erro
        if(!$confirma):

            // define informacoes da pagina
            session()->setFlashdata('msgType','danger');
            session()->setFlashdata('msg','Erro ao excluir Cliente.');
            return redirect()->back();

        // excluiu cliente com sucesso
        else:

            // define informacoes da pagina
            session()->setFlashdata('msgType','success');
            session()->setFlashdata('msg','Cliente Excluído com sucesso.');
            return redirect()->back();

        endif;
    }



    /**
     * PAINEL} Exibe informações do cliente
     * 
     * @param int $id_cliente
     * @return string
     */
    public function painel(int $id_cliente) : string {

        // carrega helper que interpreta senha na url
        helper('funcoes');

        // Informações do cliente
        $clienteModel = new ClienteModel();
        $dadosCliente = $clienteModel->find($id_cliente);

        // detalhe
        $clienteDetalheModel = new ClienteDetalheModel();
        $dadosClienteDetalhe = $clienteDetalheModel->where('id_cliente',$id_cliente)->find();
        
        // pedido
        $clientePedidoModel = new ClientePedidoModel();
        $clientePedidoModel->select('produto.*,cliente_pedido.*');
        $clientePedidoModel->where('id_cliente',$id_cliente);
        $clientePedidoModel->join('produto', 'produto.id_produto = cliente_pedido.id_produto');
        $dadosClientePedido = $clientePedidoModel->withDeleted()->find();

        // fatura
        $faturaModel = new FaturaModel();
        $faturaModel->where('id_cliente',$id_cliente);
        $faturaModel->orderBy('id_fatura','DESC');
        $dadosFatura = $faturaModel->find();

        // NFe
        $contabilModel = new ContabilModel();
        $contabilModel->select('id_contabil_s,id_fatura,con_status');
        $dadosNfe = $contabilModel->findAll();

        // orcamentos
        $orcamentoModel = new OrcamentoModel();
        $orcamentoModel->where('id_cliente',$id_cliente);
        $dadosOrcamento = $orcamentoModel->findAll();

        // fatura aberta
        $dadosFaturaAberta = [];

        // fatura fechada
        $dadosFaturaFechada = [];

        // Constroi fatura aberta / paga hoje / fatura fechada / inclui nfe à array
        foreach($dadosFatura as $fatura):

            // encontra fatura.id_fatura na array $Nfe[][id_fatura]
            $found_column = array_column($dadosNfe, 'id_fatura');
            $found_key = array_search($fatura['id_fatura'], $found_column);

            // nomeia status
            switch($fatura['fat_status']):
                case 0: $fatura['fat_status'] = 'Pago';
                    break;
                case 1: $fatura['fat_status'] = 'Aberto';
                    break;
                case 2: $fatura['fat_status'] = 'Cancelado';
                    break;
                default : $fatura['fat_status'] = $fatura['fat_status'];
                    break;
            endswitch;

            // fatura aberta
            if($fatura['fat_status'] == "Aberto"):
                if($found_key):
                    $fatura['id_contabil_s']    = $dadosNfe[$found_key]['id_contabil_s'];
                    $fatura['con_status']       = $dadosNfe[$found_key]['con_status'];
                else:
                    $fatura['id_contabil_s']    = NULL;
                    $fatura['con_status']       = NULL;
                endif;
                $dadosFaturaAberta[] = $fatura;
            // fatura fechada
            else:
                if($found_key):
                    $fatura['id_contabil_s'] = $dadosNfe[$found_key]['id_contabil_s'];
                    $fatura['con_status'] = $dadosNfe[$found_key]['con_status'];
                else:
                    $fatura['id_contabil_s']    = NULL;
                    $fatura['con_status']       = NULL;
                endif;
                $dadosFaturaFechada[] = $fatura;
            endif;
            
        endforeach;

        // zpro
        $zproModel = new ClienteZproModel();
        $zproModel->where('id_clientezpro', $dadosCliente['id_cliente']);
        $zpro = $zproModel->first();
        $zproModel->resetQuery();

        // resultado na pagina
        $dados = [
            'TituloPagina'      => 'Painel | Página principal',
            'cliente'           => $dadosCliente,
            'clienteDetalhe'    => $dadosClienteDetalhe,
            'clientePedido'     => $dadosClientePedido,
            'faturaAberta'      => $dadosFaturaAberta,
            'faturaFechada'     => $dadosFaturaFechada,
            'listagemOrcamento' => $dadosOrcamento,
            'zpro'              => $zpro,
        ];
        return view('admin/cliente/cliente-painel',$dados);
    }



    /**
     * AVATAR} Exclui imagem do avatar
     * 
     * @param int $id_cliente
     * @return object
     */
    public function avatarexcluir($id_cliente) : object{

        // seleciona a imagem
        $clienteModel = new ClienteModel();
        $dadosClienteModel = $clienteModel->where('id_cliente',$id_cliente)->first();
        
        // remove da base de dados
        $dados = [
            'cli_avatar'    => '',
        ];
        $sucesso = $clienteModel->update($id_cliente, $dados);

        // remove da pasta
        unlink('public/'.$dadosClienteModel['cli_avatar']);

        // redireciona
        session()->setFlashdata('msgType','success');
        session()->setFlashdata('msg','Avatar excluido com sucesso');

        return redirect()->to('cliente/painel/'.$id_cliente);
        

    }



    //-----------------------------------------------------------------[ MAILER ]
	//constroi a secao que envia emails
	//
    /**
     * BEMVINDO} Envia email de boas-vindas com os dados da conta
     * 
     * @param int $id_pedido
     * @return object
     */
	public function bemvindo($uid){

        //listando informações do ClienteProduto
        $clienteModel = new ClienteModel();
        $clientePedidoModel = new ClientePedidoModel();

        $clientePedidoModel->join('cliente','cliente.id_cliente = cliente_pedido.id_cliente');
        $clientePedidoModel->join('produto','produto.id_produto = cliente_pedido.id_produto');
        $clientePedidoModel->where('id_clientepedido',$uid);
        $Inds = $clientePedidoModel->first();

        //encapsula informacoes
        $id_cliente = $Inds['id_cliente'];
        $thisdata = $Inds;
        switch($thisdata['cliped_pagamentoperiodo']):
            case '1':
            $thisdata['cliped_pagamentoperiodo'] = "Anual";
            break;
            case '2':
            $thisdata['cliped_pagamentoperiodo'] = "Semestral";
            break;
            case '4':
            $thisdata['cliped_pagamentoperiodo'] = "Trimestral";
            break;
            case '8':
            $thisdata['cliped_pagamentoperiodo'] = "Mensal";
            break;
        endswitch;

        // incia serviço de email
        $email = \Config\Services::email();
        $config = [
            'protocol'  => 'smtp',
            'SMTPHost'  => 'mail.nucleodenegociosweb.com.br',
            'SMTPUser'  => 'adm@nucleodenegociosweb.com.br',
            'SMTPPass'  => 'lucr4tivo!',
            'SMTPPort'  => '587',
            'wordWrap'  => true,
            'mailType'  => 'html',
        ];
        $email->initialize($config);
        $email->setFrom('adm@nucleodenegociosweb.com.br', 'Núcleo de Negócios WEB . Carol');
        $email->setTo(array($thisdata['cli_email']));
        $email->setSubject('[Nova Conta] Informações e Senhas de Acesso');
        $email->setAltMessage('Uma novo pedido foi cadastrado em seu nome e os dados podem ser acessados através do site www.nucleodenegociosweb.com.br/cliente ou solicite por whatsapp/e-mail. \n\n Equipe Núcleo de Negócios WEB');

        // PREVINE ENVIO DE EMAIL EM LOCALHOST
        // 1 = envia email
        // 2 = gera layout do email na tela, nao envia email
        // 3 = imprime na tela, não envia email nem layout
        if($_SERVER['CI_ENVIRONMENT'] == 'development'):
            $flag = 2;
            // $flag = 3;
        else:
            $flag = 1;
        endif;
        
        // se 1 : envia email
        if($flag == 1){
            // define template para o email
            $template = view('admin/emailer/create-account', $thisdata);
            $email->setMessage($template);

            // envia mensagem
            if ($email->send()):
                session()->setFlashdata('msgType','info');
                session()->setFlashdata('msg','Email enviado.');
                return redirect()->to('cliente/painel/'.$id_cliente);
            else:
                session()->setFlashdata('msgType','info');
                session()->setFlashdata('msg','Email enviado.');
                return redirect()->back();
            endif;

        // se 2 : retorna view na tela
        }elseif($flag == 2){
            return view('admin/emailer/create-account', $thisdata);

        // se 3 : retorna variaveis na tela
        }else{

            // mostra variaveis
            echo '<hr>';
            echo '<p>var_dump:</p>';
            var_dump($Inds);
            echo '<hr>';
            die;

        }
       
        
    }



    /**
     * ATIVAÇÃO ASAAS} Chama api asaas e registra novo cliente
     * 
     * @param int $id_cliente
     * @return string
     */
    public function ativar(int $id_cliente){

        // dados do cliente
        $clienteModel = new ClienteModel();
        $clienteModel->where('id_cliente',$id_cliente);
        $dadosCliente = $clienteModel->first();

        // inicia library
        $asaas = new Asaas();

        // Verificação de duplicidade de cliente
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, "https://www.asaas.com/api/v3/customers?cpfCnpj=".$dadosCliente['cli_cpfcnpj']."&offset=&limit=");
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, TRUE);
        curl_setopt($ch, CURLOPT_HEADER, FALSE);
        curl_setopt($ch, CURLOPT_HTTPHEADER, array(
        "access_token: " .$asaas->chave()
        ));
        $response = curl_exec($ch);
        curl_close($ch);
        $validacao = json_decode($response, true);

        // Cadastro duplicado atualiza base, redireciona e infoma na tela
        if(!empty($validacao['data'])):
            $id_asaas = $validacao['data'][0]['id'];
            $clienteModel->set('id_asaas', $id_asaas);
            $clienteModel->where('id_cliente', $id_cliente);
            $atualiza = $clienteModel->update();
            if($atualiza):
                session()->setFlashdata('msgType','warning');
                session()->setFlashdata('msg','Usuário já ativo no sistema. Base de dados atualizada com id: '.$id_asaas);
            else:
                session()->setFlashdata('msgType','danger');
                session()->setFlashdata('msg','Usuário já ativo no sistema. Base de dados não conseguiu atualizar com id: '.$id_asaas);
            endif;
            return redirect()->back();
        endif;

        // Informações para cadastro
        $dadosCliente = [
        'name'                  => $dadosCliente['cli_nome'],
        'cpfCnpj'               => $dadosCliente['cli_cpfcnpj'],
        'email'                 => $dadosCliente['cli_email'],
        'phone'                 => $dadosCliente['cli_telefone'],
        'mobilePhone'           => $dadosCliente['cli_telefone'],
        'address'               => $dadosCliente['cli_endereco'],
        'addressNumber'         => $dadosCliente['cli_endereco_numero'],
        'complement'            => $dadosCliente['cli_endereco_compl'],
        'province'              => $dadosCliente['cli_bairro'],
        'postalCode'            => $dadosCliente['cli_cep'],
        'externalReference'     => $id_cliente,
        'notificationDisabled'  => true,
        // 'additionalEmails'      => $dadosCliente[''],
        // 'municipalInscription'  => $dadosCliente['cli_insc_municipal'],
        'stateInscription'      => $dadosCliente['cli_insc_estadual'],
        ];

        // Registra Cliente
        $ch = curl_init();
        // curl_setopt($ch, CURLOPT_URL, "https://private-anon-7525df9a65-asaasv3.apiary-proxy.com/api/v3/customers");
        curl_setopt($ch, CURLOPT_URL, "https://www.asaas.com/api/v3/customers");
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, TRUE);
        curl_setopt($ch, CURLOPT_HEADER, FALSE);
        curl_setopt($ch, CURLOPT_POST, TRUE);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($dadosCliente));
        curl_setopt($ch, CURLOPT_HTTPHEADER, array(
        "Content-Type: application/json",
        "access_token: ". $asaas->chave(),
        ));
        $response = curl_exec($ch);
        curl_close($ch);
        $retorno = json_decode($response, true);

        // previne erro
        if(isset($retorno['errors'])):
            session()->setFlashdata('msgType','danger');
            session()->setFlashdata('msg',$retorno['errors'][0]['description']);
            return redirect()->back();
        endif;
        
        // salva
        $id_asaas = $retorno['id'];
        $clienteModel->set('id_asaas', $id_asaas);
        $clienteModel->where('id_cliente', $id_cliente);
        $atualiza = $clienteModel->update();

        // retorna
        if($atualiza):
            session()->setFlashdata('msgType','success');
            session()->setFlashdata('msg','Usuário criado e Base de dados atualizada com id: '.$id_asaas);
        else:
            session()->setFlashdata('msgType','danger');
            session()->setFlashdata('msg','Tente novamente. Usuário criado mas base não atualizou com id: '.$id_asaas);
        endif;
        return redirect()->back();
    }



    /**
     * ATIVAÇÃO Z-Pró}
     * Chama api e registra novo cliente
     * 
     * @param int $id_cliente
     * @return string
     */
    public function ativarZpro(int $id_cliente){


        $rules = [
            'clizpr_apiUrl' =>[
                'rules'     => 'required',
                'errors'    => [
                    'required'  => 'Preencha o campo',
                ],
            ],
            'clizpr_token' =>[
                'rules'     => 'required',
                'errors'    => [
                    'required'  => 'Preencha o campo',
                ],
            ],
            'maxUsers' =>[
                'rules'     => 'required',
                'errors'    => [
                    'required'  => 'Preencha o campo',
                ],
            ],
            'maxConnections' =>[
                'rules'     => 'required',
                'errors'    => [
                    'required'  => 'Preencha o campo',
                ],
            ],
            'email' =>[
                'rules'     => 'required|valid_email',
                'errors'    => [
                    'required'  => 'Preencha o campo',
                ],
            ],
            'password' =>[
                'rules'     => 'required',
                'errors'    => [
                    'required'  => 'Preencha o campo',
                ],
            ],
            'userName' =>[
                'rules'     => 'required',
                'errors'    => [
                    'required'  => 'Preencha o campo',
                ],
            ],
            'identity' =>[
                'rules'     => 'required',
                'errors'    => [
                    'required'  => 'Preencha o campo',
                ],
            ],
            'trial' =>[
                'rules'     => 'required',
                'errors'    => [
                    'required'  => 'Preencha o campo',
                ],
            ],
            'trial_period' =>[
                'rules'     => 'permit_empty',
            ],
        ];

        // Validação apresentou erro
        if(!$this->validate($rules)){

            // recarrega a página
            return redirect()->back()->to('cliente/painel/'.$id_cliente.'#nav-zpro')->withInput();

        // Validação executou com sucesso
        }else{

            // recebe dados do formulário
            $formulario = $this->request->getPost();

            // evita erro
            if( $formulario['trial'] == 'enabled' ):
                if( $formulario['trialPeriod'] == "" ): 

                    // redireciona
                    session()->setFlashdata('msgType','danger');
                    session()->setFlashdata('msg','Defina um número de dias de teste para o trial');
                    return redirect()->back()->to('cliente/painel/'.$id_cliente.'#nav-zpro')->withInput();
                
                endif;
            endif;

            // dados do cliente
            $clienteModel = new ClienteModel();
            $clienteModel->where('id_cliente',$id_cliente);
            $cliente = $clienteModel->first();
            $clienteModel->resetQuery();

            // dados zpro
            $clienteZproModel = new ClienteZproModel();
            $clienteZproModel->where('id_clientezpro', $cliente['id_clientezpro']);
            $zpro = $clienteZproModel->first();
            $clienteZproModel->resetQuery();

            // dados se zpro se retornar vazio então cria na base
            if( !$zpro ):
                
                // cria um server
                $dados = [
                    'clizpr_apiUrl'     => $formulario['clizpr_apiUrl'],
                    'clizpr_token'      => $formulario['clizpr_token'],
                    'status'            => $formulario['status'],
                    'name'              => $formulario['name'],
                    'maxUsers'          => $formulario['maxUsers'],
                    'maxConnections'    => $formulario['maxConnections'],
                    'acceptTerms'       => true, // aceito os termos
                    'email'             => $formulario['email'],
                    'password'          => $formulario['password'],
                    'userName'          => $formulario['userName'],
                    'identity'          => $formulario['identity'],
                    'profile'           => 'admin', // tipo do perfil criado
                    'trial'             => $formulario['trial'],
                    'trialPeriod'       => $formulario['trialPeriod'],  
                ];
                $id_zpro = $zproModel->insert( $dados );
                $zproModel->resetQuery();

                // atualiza cliente
                $clienteModel->set('id_clientezpro', $id_zpro);
                $clienteModel->where('id_cliente', $cliente['id_cliente']);
                $clienteModel->update();
                $clienteModel->resetQuery();

                // seleciona zpro
                $zproModel->where('id_clientezpro', $id_zpro );
                $zpro = $zproModel->first();
                $zproModel->resetQuery();
            endif;


            // se existe a id_tenant criada no servidor
            // pesquisa na apiZpro se existe e atualiza status
            $tenantImportada = false;
            if( $zpro['id_tenant'] != '' ):
                $curl = curl_init();
                curl_setopt_array($curl, array(
                CURLOPT_URL => rtrim($zpro['clizpr_apiUrl'], '/') . '/tenantApiShowTenant',
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_ENCODING => '',
                CURLOPT_MAXREDIRS => 10,
                CURLOPT_TIMEOUT => 0,
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
                CURLOPT_CUSTOMREQUEST => 'POST',
                CURLOPT_POSTFIELDS =>'{
                "id": ' . $zpro['id_tenant'] . '
                }',
                CURLOPT_HTTPHEADER => array(
                    'Content-Type: application/json',
                    'Authorization: Bearer ' . $zpro['clizpr_token']
                ),
                ));
                $response = curl_exec($curl);
                curl_close($curl);
                $tenantImportada = json_decode($response, true);

                // se exisitir atualiza na base de dados
                if( isset( $tenantImportada['tenant'][0]['status'] ) && $tenantImportada['tenant'][0]['status'] != $zpro['status'] ){
                    $clienteZproModel->set('status', $tenantImportada['tenant'][0]['status']);
                    $clienteZproModel->where('id_clientezpro', $zpro['id_clientezpro']);
                    $clienteZproModel->update();
                    $clienteZproModel->resetQuery();
                }

            endif;

            // se id_tenant não existir no servidor ou se não existir na api
            // requer a criação na apiZpro e salva id_tenant na base de dados
            $tenantCriada = false;
            if( !$tenantImportada ):

                $criaZproDados = [
                    'status'            => $zpro['status'],
                    'name'              => $zpro['name'],
                    'maxUsers'          => $zpro['maxUsers'],
                    'maxConnections'    => $zpro['maxConnections'],
                    'acceptTerms'       => $zpro['acceptTerms'],
                    'email'             => $zpro['email'],
                    'password'          => $zpro['password'],
                    'userName'          => $zpro['userName'],
                    'identity'          => $zpro['identity'],
                    'profile'           => $zpro['profile'],
                    'trial'             => $zpro['trial'],
                    'trialPeriod'       => $zpro['trialPeriod'],
                ];
            
                // Registra tenant
                $curl = curl_init();
                curl_setopt_array($curl, array(
                CURLOPT_URL => rtrim($zpro['clizpr_apiUrl'], '/') . '/tenantApiStoreTenant',
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_ENCODING => '',
                CURLOPT_MAXREDIRS => 10,
                CURLOPT_TIMEOUT => 0,
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
                CURLOPT_CUSTOMREQUEST => 'POST',
                CURLOPT_POSTFIELDS => json_encode($criaZproDados),
                CURLOPT_HTTPHEADER => array(
                    'Content-Type: application/json',
                    'Authorization: Bearer ' . $zpro['clizpr_token']
                ),
                ));
                $response = curl_exec($curl);
                curl_close($curl);
                $tenantCriada = json_decode($response, true);

                // salva id tenant criada
                if( $tenantCriada ):
                    $clienteZproModel->set('id_tenant', $tenantCriada['tenant']['id']);
                    $clienteZproModel->where('id_clientezpro', $zpro['id_clientezpro']);
                    $clienteZproModel->update();
                    $clienteZproModel->resetQuery();
                else:
                    session()->setFlashdata('msgType','danger');
                    session()->setFlashdata('msg','Tente novamente.');
                endif;


            endif;
           

            // retorna
            session()->setFlashdata('msgType','success');
            session()->setFlashdata('msg','Api rodou com sucesso');
            return redirect()->back()->to('cliente/painel/'.$id_cliente.'#nav-zpro');

        }
    }



    /**
     * ALTERA O STATUS DA TENANT }
     * Chama api e altera o status da id_tenant
     */
    public function alteraStatus(int $id_cliente){

        // dados do cliente
        $clienteModel = new ClienteModel();
        $clienteModel->where('id_cliente',$id_cliente);
        $cliente = $clienteModel->first();
        $clienteModel->resetQuery();

        // dados zpro
        $clienteZproModel = new ClienteZproModel();
        $clienteZproModel->where('id_clientezpro', $cliente['id_clientezpro']);
        $zpro = $clienteZproModel->first();
        $clienteZproModel->resetQuery();

        if( $zpro['status'] == 'active' ):
            $statusToggle = 'inactive';
        else:
            $statusToggle = 'active';
        endif;

        // requer alteração de status
        $curl = curl_init();
        curl_setopt_array($curl, array(
        CURLOPT_URL => rtrim($zpro['clizpr_apiUrl'], '/') . '/tenantApiUpdateTenant',
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_ENCODING => '',
        CURLOPT_MAXREDIRS => 10,
        CURLOPT_TIMEOUT => 0,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
        CURLOPT_CUSTOMREQUEST => 'POST',
        CURLOPT_POSTFIELDS =>'{
        "status": "' . $statusToggle . '",
        "identity": "' . $zpro['identity'] . '"
        }',
        CURLOPT_HTTPHEADER => array(
            'Content-Type: application/json',
            'Authorization: Bearer ' . $zpro['clizpr_token']
        ),
        ));
        $response = curl_exec($curl);
        curl_close($curl);
        $retorno= json_decode($response, true);

        // atualiza retorno na base de dados
        if( isset($retorno['status'])):
            $clienteZproModel->set('status', $retorno['status']);
            $clienteZproModel->where('id_clientezpro', $zpro['id_clientezpro']);
            $clienteZproModel->update();
            $clienteZproModel->resetQuery();
        endif;

        // retorna
        if($retorno):
            session()->setFlashdata('msgType','success');
            session()->setFlashdata('msg','Api rodou');
        else:
            session()->setFlashdata('msgType','danger');
            session()->setFlashdata('msg','Tente novamente, ocorreu algo.');
        endif;
        session()->setFlashdata('msg', json_encode($retorno) );

        return redirect()->back()->to('cliente/painel/'.$id_cliente.'#nav-zpro');

    }



    /**
     *  LEMBRETE WHATSAPP} Envia notificação para o whatsapp do cliente
     * 
     * @param $id_cliente
     * @return 
     */
    public function lembreteWhats($id_cliente) {

        helper('funcoes');

        // seleciona o telefone do cliente
        $clienteModel = new ClienteModel();
        $clienteModel->where('id_cliente', $id_cliente);
        $cliente = $clienteModel->first();

        // seleciona informações para o bot
        $organizacaoModel = new OrganizacaoModel();
        $organizacaoModel->select('org_botUrl,org_botAuth,org_botHash');
        $organizacao = $organizacaoModel->first();
        
        // primeiro nome
        $cliente['cli_nome'] = explode(' ',$cliente['cli_nome']);

        // senha codificada para url
        $cliente['cli_senha'] = urlencode(urilencode($cliente['cli_senha']));

        // BOT > ENVIA MENSAGEM TESTE
        $botZdg = new Zdg($organizacao['org_botUrl'],$organizacao['org_botAuth'],$organizacao['org_botHash']);
        $botZdg->numero($cliente['cli_telefone']);
        
        $message1 = '*Carol:*
Oi '. $cliente['cli_nome'][0] .'

Espero que esteja tudo bem.';

        // dispara a mensagem 1
        $botZdg->mensagem($message1);
        $zdgRetorno = $botZdg->sendMessage();
        var_dump($zdgRetorno);
        
        sleep(1);

        $message2 = '*Carol:*
Informo que existe uma nova informação em seu painel de cliente.';

        //dispara a mensagem 2
        $botZdg->mensagem($message2);
        $zdgRetorno = $botZdg->sendMessage();
        var_dump($zdgRetorno);

        sleep(3);

        $message3 = '*Carol:*
Confira acessando através do link:
https://www.nucleodenegociosweb.com.br/crm/loginCliente/fatura/'.$cliente['id_cliente'].':aois:'.$cliente['cli_senha'];

        //dispara a mensagem 3
        $botZdg->mensagem($message3);
        $zdgRetorno = $botZdg->sendMessage();
        var_dump($zdgRetorno);

        // retorna
        session()->setFlashdata('msgType','success');
        session()->setFlashdata('msg','Rodou script');
        return redirect()->to('cliente/painel/'.$cliente['id_cliente']);

    }




}